/*************************************************************************
* Copyright (C) 2013 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

/*
//
//  Purpose:
//     Cryptography Primitive.
//     RSA Functions
//
//  Contents:
//        ippsRSA_Encrypt()
//
*/

#include "owncp.h"
#include "pcpbn.h"
#include "pcpngrsa.h"
#include "pcpngrsamethod.h"

/*F*
// Name: ippsRSA_Encrypt
//
// Purpose: Performs RSA Encryprion
//
// Returns:                   Reason:
//    ippStsNullPtrErr           NULL == pKey
//                               NULL == pPtxt
//                               NULL == pCtxt
//                               NULL == pBuffer
//
//    ippStsContextMatchErr     !RSA_PUB_KEY_VALID_ID()
//                              !BN_VALID_ID(pPtxt)
//                              !BN_VALID_ID(pCtxt)
//
//    ippStsIncompleteContextErr public key is not setup
//
//    ippStsOutOfRangeErr        pPtxt >= modulus
//                               pPtxt <0
//
//    ippStsSizeErr              BN_ROOM(pCtxt) is not enough
//
//    ippStsNoErr                no error
//
// Parameters:
//    pPtxt          pointer to the plaintext
//    pCtxt          pointer to the ciphertext
//    pKey           pointer to the key context
//    pScratchBuffer pointer to the temporary buffer
*F*/
/* clang-format off */
IPPFUN(IppStatus, ippsRSA_Encrypt, (const IppsBigNumState* pPtxt,
                                    IppsBigNumState* pCtxt,
                                    const IppsRSAPublicKeyState* pKey,
                                    Ipp8u* pBuffer))
/* clang-format on */
{
    IPP_BAD_PTR2_RET(pKey, pBuffer);
    IPP_BADARG_RET(!RSA_PUB_KEY_VALID_ID(pKey), ippStsContextMatchErr);
    IPP_BADARG_RET(!RSA_PUB_KEY_IS_SET(pKey), ippStsIncompleteContextErr);

    IPP_BAD_PTR1_RET(pPtxt);
    IPP_BADARG_RET(!BN_VALID_ID(pPtxt), ippStsContextMatchErr);
    IPP_BADARG_RET(BN_NEGATIVE(pPtxt), ippStsOutOfRangeErr);
    IPP_BADARG_RET(0 <= cpCmp_BNU(BN_NUMBER(pPtxt),
                                  BN_SIZE(pPtxt),
                                  MOD_MODULUS(RSA_PUB_KEY_NMONT(pKey)),
                                  MOD_LEN(RSA_PUB_KEY_NMONT(pKey))),
                   ippStsOutOfRangeErr);

    IPP_BAD_PTR1_RET(pCtxt);
    IPP_BADARG_RET(!BN_VALID_ID(pCtxt), ippStsContextMatchErr);
    IPP_BADARG_RET(BN_ROOM(pCtxt) < BITS_BNU_CHUNK(RSA_PUB_KEY_BITSIZE_N(pKey)), ippStsSizeErr);

    {
        BNU_CHUNK_T* pScratchBuffer =
            (BNU_CHUNK_T*)(IPP_ALIGNED_PTR(pBuffer, (int)sizeof(BNU_CHUNK_T)));
        gsRSApub_cipher(pCtxt, pPtxt, pKey, pScratchBuffer);
        return ippStsNoErr;
    }
}
